<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use App\Services\ShopifyService;

class EnsureOnlineStoreChannel
{
    public function handle(Request $request, Closure $next)
    {
        // Get the authenticated user
        $user = Auth::user();

        // Check if user is authenticated
        if (!$user) {
            return redirect()->route('error')->with('message', 'Authentication required.');
        }

        // Fetch shop domain and access token (ensure correct column names based on your DB)
        if (isset($user->shop_domain)) {
            $shopDomain = $user->shop_domain;
        } else {
            $shopDomain = $user->name;  // You may adjust this based on your actual database structure
        }

        if (isset($user->password)) {
            $accessToken = $user->password;
        } else {
            $accessToken = null;
        }

        // Check if both shopDomain and accessToken are present
        if (!$shopDomain || !$accessToken) {
            Log::warning('Missing shop domain or access token', [
                'user_id' => $user->id,
                'shopDomain' => $shopDomain,
                'accessTokenPresent' => $accessToken !== null,
            ]);
            return redirect()->route('error')->with('message', 'Missing Shopify store credentials.');
        }

        // Instantiate the ShopifyService to check if "online-store" channel exists
        $shopifyService = new ShopifyService();

        // Check if the online store channel is enabled for the shop
        if (!$shopifyService->hasOnlineStoreChannel($shopDomain, $accessToken)) {
            return redirect()->route('error')->with('message', 'Online Store sales channel is not enabled for this shop.');
        }

        // Proceed with the next middleware or the request itself
        return $next($request);
    }
}
